/*
 * cSimNetLayer.cpp
 *
 * THE simulated network layer.
 */
#include "cSimNetLayer.h"
#include "cSimNetLayerParam.h"
#include "cSimNode.h"
#include "cSimEngine.h"
#include "Util/gError.h"
#include "Util/cRegisterList.h"
#include "Util/cException.h"
#include "Protocol Stack/cProtocolStack.h"
#include "Endpoint/cEndpointFactory.h"

#include <memory.h>
#include <malloc.h>
#include <stdio.h>

/*
 * cSimNetLayer::Deliver()
 *
 * Purpose:	Delivery interface function.
 * IN:		sender		-> The sender of the msg
 *			buffer		-> The buffer delivered up.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The layer is ready to rock.
 * Return:	true if success, else false.
 */
bool cSimNetLayer::Deliver(cEndpoint* sender, cMsgBuffer* buffer, int messageType)
{
	cIterator*	iter;
	cDeliver*	deliver;

	// Deliver to callbacks.
	sender->AddRef();
	iter = mRegisterList->GetIterator();
	while(!iter->Done())
	{
		deliver = (cDeliver *)iter->GetData();
		deliver->Deliver(sender, buffer, messageType);
		iter->GetNext();
	}
	sender->Release();

	return true;
}

/*
 * cSimNetLayer::Init()
 *
 * Purpose:	Initializes the sim layer.
 * IN:		layerBelow	-> The layer below this layer.
 *			param		-> The parameters intended for this layer.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The layer is ready to rock.
 * Return:	true if success, else false.
 */
bool cSimNetLayer::Init(cLayer* layerBelow, cParam* param)
{
	cSimNetLayerParam*	nParam = (cSimNetLayerParam *)param;

	// Init our pointer to the protocol stack.
	mProtocolStack = param->mProtocolStack;
	mEngine = nParam->mEngine;

	// Set up register list.
	mRegisterList = new cRegisterList(1);
	if(!mRegisterList)
	{
		return false;
	}

	// Set up the error callback list.
	mErrorCallbackList = new cRegisterList(1); 
	if(!mErrorCallbackList)
	{
		return false;
	}

	// Create network event and register it with protocolstack
	mNetEvent = CreateEvent(NULL, 
							TRUE,	// manual reset
							FALSE,	// not signaled
							NULL);
	if(mNetEvent == NULL) { return false; }
	mProtocolStack->AddEvent(mNetEvent, this, NULL);

	// Get local endpoint and add it to pstack
	if(!mEngine->GetFreshEndpoint(mNetEvent, &mMyEndpoint))
	{
		gError("Sim engine ran out of endpoints.", __LINE__, __FILE__);
		return false;
	}
	mProtocolStack->AddLocalAddress((cEndpoint *)mMyEndpoint);

	// Copy layer info from below
	mLayerBelow = layerBelow;
	if(layerBelow)
	{
		cLayerInfo* belowInfo;
		belowInfo = mLayerBelow->GetLayerInfo();
		mLayerInfo = *belowInfo;
	}
	mLayerInfo.SetLayerBelow(mLayerBelow);
	mLayerInfo.AddHeaderSize(0);

	// Register with layer below
	if(mLayerBelow)
	{
		if(!layerBelow->RegisterDeliverCallback(&mLayerBelowHandle, this))
		{
			gError("Unable to register with layer below.", __LINE__, __FILE__);
			return false;
		}
	}
	return true;
}

/*
 * cSimNetLayer::Cleanup()
 *
 * Purpose:	Cleans up the network layer.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The layer is no longer useable.
 * Return:	true if success, else false.
 */
bool cSimNetLayer::Cleanup()
{
	if(mRegisterList)		{ delete mRegisterList; }
	if(mErrorCallbackList)	{ delete mErrorCallbackList; }
	if(mLayerBelow)
	{
		mLayerBelow->UnregisterDeliverCallback(mLayerBelowHandle);
	}
	return true;
}

/*
 * cSimNetLayer::Send()
 *
 * Purpose:	Prints out the msg to send and passes it to layer below.
 * IN:		dest		-> The message destination.
 *			buffer		-> the actual message.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The layer is ready to rock.
 * Return:	true if success, else false.
 */
bool cSimNetLayer::Send(cGroup* dest, cMsgBuffer* buffer, int messageType)	
{
	cIterator* iter;
	cEndpoint* ep;

	iter = dest->GetIterator();
	while(!iter->Done())
	{
		ep = (cEndpoint *)iter->GetData();
		if(ep->GetType() == ENDPOINT_TYPE_SIM)
		{
			mEngine->Send(mMyEndpoint, (cSimEndpoint *)ep, buffer, messageType);
		}
		iter->GetNext();
	}

	return true; 
}

/*
 * cSimNetLayer::RegisterDeliverCallback()
 *
 * Purpose:	Registers the deliver callback.
 * IN:		callback	-> The callback function to register.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The callback function is registered.
 * Return:	The handle used for unregistering.
 */
bool cSimNetLayer::RegisterDeliverCallback(cHandle* handle, cDeliver* callback) 
{ 
	handle->mLayer = this;
	return mRegisterList->AddObject(handle, (cObject *)callback);
}

/*
 * cSimNetLayer::UnregisterDeliverCallback()
 *
 * Purpose:	Unregisters the deliver callback.
 * IN:		handle	-> The handle that was received at register.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The callback function is removed.
 * Return:	true if success, else false.
 */
bool cSimNetLayer::UnregisterDeliverCallback(cHandle handle)	
{ 
	if(handle.mLayer == this)
	{
		return mRegisterList->RemoveObject(handle);
	}
	else if(mLayerBelow)
	{
		return mLayerBelow->UnregisterDeliverCallback(handle);
	}
	else
	{
		return false;
	}
}

/*
 * cSimNetLayer::RegisterErrorCallback()
 *
 * Purpose:	Registers the asynchronous error callback.
 * IN:		callback	-> The callback function to register.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The callback function is registered.
 * Return:	The handle used for unregistering.
 */
bool cSimNetLayer::RegisterErrorCallback(cHandle* handle, cErrorCallback* callback)
{ 
	handle->mLayer = this;
	return mErrorCallbackList->AddObject(handle, (cObject *)callback);
}

/*
 * cSimNetLayer::UnregisterErrorCallback()
 *
 * Purpose:	Unregisters the error callback.
 * IN:		handle	-> The handle that was received at register.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The callback function is removed.
 * Return:	true if success, else false.
 */
bool cSimNetLayer::UnregisterErrorCallback(cHandle handle)
{ 
	if(handle.mLayer == this)
	{
		return mErrorCallbackList->RemoveObject(handle);
	}
	else if(mLayerBelow)
	{
		return mLayerBelow->UnregisterErrorCallback(handle);
	}
	else
	{
		return false;
	}
}

/*
 * cSimNetLayer::Callback()
 *
 * Purpose: Callback on socket receive.
 * IN:		eventHandle -> The handle for the triggered event.
 *			param		-> The socket to receive on.
 * OUT:		-
 * Return:  -
 */
void cSimNetLayer::Callback(HANDLE eventHandle, void* param)
{
	cMsgBuffer*		msg;
	cSimEndpoint*	sep;
	unsigned int	type;

	ResetEvent(mNetEvent);	// Set event to non-signaled
	while(mEngine->GetMessage(mMyEndpoint, &sep, &msg, &type))
	{
//		sep->AddRef();
//		msg->AddRef();
		Deliver((cEndpoint *)sep, msg, type);
		sep->Release();
		msg->Release();
	}
}

/*
 * cSimNetLayer::_DeliverErrorCallback()
 *
 * Purpose:	Delivers asynch errors to any registrees.
 * IN:		obj		-> the object parameter
 *			type	-> the type parameter
 * OUT:		-
 * Return:	The local ip address.
 */
bool cSimNetLayer::_DeliverErrorCallback(cObject* obj, unsigned int type)
{
	cIterator*		iter;
	cErrorCallback* callback;

	iter = mErrorCallbackList->GetIterator();
	while(!iter->Done())
	{
		callback = (cErrorCallback *)iter->GetData();
		callback->ErrorCallback(obj, type);
		iter->GetNext();
	}
	return true;
}

bool	cSimNetLayer::RegisterViewCallback(cHandle* handle, cView* callback)
{
	if(mLayerBelow)
	{
		return mLayerBelow->RegisterViewCallback(handle, callback);
	}
	else
	{
		return false;
	}
}
bool	cSimNetLayer::UnregisterViewCallback(cHandle handle)
{
	if(mLayerBelow)
	{
		return mLayerBelow->UnregisterViewCallback(handle);
	}
	else
	{
		return false;
	}
}